<?php
require_once 'config.php';
require_once 'database.php';

class TelegramBot {
    private $apiToken;
    private $update;
    private $message;
    private $chatId;
    private $userId;
    private $text;
    private $firstName;
    private $lastName;
    private $username;
    private $db;
    private $userStates = [];

    public function __construct() {
        $this->apiToken = BOT_TOKEN;
        $this->update = json_decode(file_get_contents("php://input"), TRUE);
        $this->db = new Database();
        
        if (isset($this->update['callback_query'])) {
            $callback = $this->update['callback_query'];
            $this->chatId = $callback['message']['chat']['id'];
            $this->userId = $callback['from']['id'];
            $this->firstName = isset($callback['from']['first_name']) ? $callback['from']['first_name'] : "نام‌ناشناس";
            $this->lastName = isset($callback['from']['last_name']) ? $callback['from']['last_name'] : "";
            $this->username = isset($callback['from']['username']) ? "@" . $callback['from']['username'] : "نام کاربری ندارد";
            $this->handleCallback($callback);
            exit;
        }
        
        if (isset($this->update['message'])) {
            $this->message = $this->update['message'];
            $this->chatId = $this->message['chat']['id'];
            $this->userId = $this->message['from']['id'];
            $this->text = $this->message['text'] ?? '';
            $this->firstName = isset($this->message['from']['first_name']) ? $this->message['from']['first_name'] : "نام‌ناشناس";
            $this->lastName = isset($this->message['from']['last_name']) ? $this->message['from']['last_name'] : "";
            $this->username = isset($this->message['from']['username']) ? "@" . $this->message['from']['username'] : "نام کاربری ندارد";
        }
    }

    public function handleUpdate() {
        if (!isset($this->message)) {
            return;
        }

        // بررسی مدیر بودن کاربر
        if ($this->isManager()) {
            $this->handleManagerCommands();
            return;
        }

        // ثبت‌نام مرحله‌ای همایش (اولویت بالا)
        if ($this->checkUserRegisterState()) {
            $this->handleUserRegisterState();
            return;
        }

        // وضعیت ارسال فیدبک
        if ($this->checkUserFeedbackState()) {
            $this->handleUserFeedbackState();
            return;
        }

        // مدیریت دستور /start
        if ($this->text === "/start") {
            $this->showUserMenu();
            return;
        }

        // بررسی عضویت در کانال‌ها
        if (!$this->checkChannelMembership()) {
            $this->handleNotMember();
            return;
        }

        // مدیریت منوی کاربری
        if ($this->text === "ارسال فیدبک") {
            $this->setUserFeedbackState();
            $this->sendMessage($this->chatId, "الان می‌توانید فیدبک خود را ارسال کنید.");
            return;
        }
        if ($this->text === "لیست همایش‌ها") {
            $this->showUserConferences();
            return;
        }

        // نمایش منوی کاربری پیش‌فرض
        $this->showUserMenu();
    }

    private function handleManagerCommands() {
        // وضعیت را از فایل بخوان
        $state = $this->getManagerState();
        if ($state && $state['state'] === 'ADD_CONFERENCE') {
            $this->handleAddConferenceState($state);
            return;
        }

        if ($this->text === "/start") {
            $this->showManagerMenu();
            return;
        }

        switch ($this->text) {
            case "📋 لیست همایش‌ها":
                $this->showConferencesList();
                break;
            case "➕ افزودن همایش":
                $state = [
                    'state' => 'ADD_CONFERENCE',
                    'step' => 'name',
                    'data' => []
                ];
                $this->setManagerState($state);
                $this->sendMessage($this->chatId, "لطفاً نام همایش را وارد کنید:");
                break;
            default:
                $this->showManagerMenu();
        }
    }

    private function showManagerMenu() {
        $keyboard = [
            'keyboard' => [
                [['text' => '📋 لیست همایش‌ها'], ['text' => '➕ افزودن همایش']]
            ],
            'resize_keyboard' => true
        ];

        $this->sendMessage(
            $this->chatId,
            "👋 سلام مدیر گرامی!\nلطفاً یکی از گزینه‌های زیر را انتخاب کنید:",
            json_encode($keyboard)
        );
    }

    private function showConferencesList() {
        $conferences = $this->db->getConferences();
        if (!$conferences) {
            $this->sendMessage($this->chatId, "❌ در حال حاضر هیچ همایشی ثبت نشده است.");
            return;
        }
        foreach ($conferences as $conference) {
            $message = "🔹 نام: {$conference['name']}\n";
            $message .= "👥 تعداد نفرات: {$conference['participants_count']}\n";
            $message .= "📅 تاریخ: {$conference['conference_date']}\n";
            $message .= "⏰ ساعت: {$conference['start_time']} تا {$conference['end_time']}\n";
            $message .= "📍 آدرس: {$conference['address']}\n";
            $message .= "🔗 لینک لوکیشن: {$conference['location_link']}\n";
            $message .= "📢 کانال: {$conference['channel_id']}\n";
            $participants = $this->db->getParticipants($conference['id']);
            $hasParticipants = count($participants) > 0;
            $keyboard = ['inline_keyboard' => []];
            if ($hasParticipants) {
                $keyboard['inline_keyboard'][] = [
                    ['text' => '📥 خروجی اکسل', 'callback_data' => 'export_' . $conference['id']]
                ];
            } else {
                $keyboard['inline_keyboard'][] = [
                    ['text' => '🗑 حذف', 'callback_data' => 'delete_' . $conference['id']],
                    ['text' => '📥 خروجی اکسل', 'callback_data' => 'export_' . $conference['id']]
                ];
            }
            $this->sendMessage(
                $this->chatId,
                $message,
                json_encode($keyboard)
            );
        }
    }

    private function getManagerState() {
        if (file_exists('manager_state.json')) {
            $json = file_get_contents('manager_state.json');
            return json_decode($json, true);
        }
        return null;
    }

    private function setManagerState($state) {
        file_put_contents('manager_state.json', json_encode($state));
    }

    private function clearManagerState() {
        if (file_exists('manager_state.json')) {
            unlink('manager_state.json');
        }
    }

    private function handleAddConferenceState($state) {
        $data = $state['data'];
        
        switch ($state['step']) {
            case 'name':
                $data['name'] = $this->text;
                $state['step'] = 'participants_count';
                $state['data'] = $data;
                $this->setManagerState($state);
                $this->sendMessage($this->chatId, "تعداد نفرات همایش را وارد کنید:");
                break;

            case 'participants_count':
                $data['participants_count'] = (int)$this->faToEn($this->text);
                $state['step'] = 'conference_date';
                $state['data'] = $data;
                $this->setManagerState($state);
                $this->sendMessage($this->chatId, "تاریخ همایش را به فرمت YYYY-MM-DD وارد کنید:");
                break;

            case 'conference_date':
                $data['conference_date'] = $this->faToEn($this->text);
                $state['step'] = 'address';
                $state['data'] = $data;
                $this->setManagerState($state);
                $this->sendMessage($this->chatId, "آدرس همایش را وارد کنید:");
                break;

            case 'address':
                $data['address'] = $this->text;
                $state['step'] = 'location_link';
                $state['data'] = $data;
                $this->setManagerState($state);
                $this->sendMessage($this->chatId, "لینک لوکیشن همایش را وارد کنید:");
                break;

            case 'location_link':
                $data['location_link'] = $this->text;
                $state['step'] = 'start_time';
                $state['data'] = $data;
                $this->setManagerState($state);
                $this->sendMessage($this->chatId, "ساعت شروع همایش را به فرمت HH:MM وارد کنید:");
                break;

            case 'start_time':
                $data['start_time'] = $this->faToEn($this->text);
                $state['step'] = 'end_time';
                $state['data'] = $data;
                $this->setManagerState($state);
                $this->sendMessage($this->chatId, "ساعت پایان همایش را به فرمت HH:MM وارد کنید:");
                break;

            case 'end_time':
                $data['end_time'] = $this->faToEn($this->text);
                $state['step'] = 'channel_id';
                $state['data'] = $data;
                $this->setManagerState($state);
                $this->showChannelSelection();
                break;

            case 'channel_id':
                global $channels;
                // جستجوی آی‌دی کانال بر اساس نام وارد شده
                $channel_id = null;
                foreach ($channels as $id => $name) {
                    if ($name === $this->text) {
                        $channel_id = $id;
                        break;
                    }
                }
                if (!$channel_id) {
                    $this->sendMessage($this->chatId, "❌ کانال انتخاب‌شده معتبر نیست. لطفاً از لیست انتخاب کنید.");
                    return;
                }
                $data['channel_id'] = $channel_id;
                try {
                    if ($this->db->addConference($data)) {
                        $this->sendMessage($this->chatId, "✅ همایش با موفقیت ثبت شد.");
                        $this->clearManagerState();
                        $this->showManagerMenu();
                    } else {
                        $this->sendMessage($this->chatId, "❌ خطا در ثبت همایش. لطفاً دوباره تلاش کنید.");
                    }
                } catch (Exception $e) {
                    error_log($e->getMessage());
                    $this->sendMessage($this->chatId, "❌ خطا در ثبت همایش: " . $e->getMessage());
                }
                break;
        }
    }

    private function showChannelSelection() {
        global $channels;
        $keyboard = ['keyboard' => [], 'resize_keyboard' => true];
        
        foreach ($channels as $id => $name) {
            $keyboard['keyboard'][] = [['text' => $name]];
        }

        $this->sendMessage(
            $this->chatId,
            "لطفاً کانال مربوط به این همایش را انتخاب کنید:",
            json_encode($keyboard)
        );
    }

    private function isManager() {
        return $this->chatId == MANAGER_CHAT_ID || $this->chatId == MANAGER_CHAT_ID_2;
    }

    private function handleStart() {
        $this->sendVoice(
            $this->chatId,
            'https://sadataalaee.com/bots/feedback/start.mp3',
            WELCOME_MESSAGE
        );
    }

    private function checkChannelMembership() {
        global $channels;
        foreach ($channels as $channelId => $channelName) {
            $getChatMemberUrl = "https://api.telegram.org/bot{$this->apiToken}/getChatMember?chat_id=$channelId&user_id={$this->userId}";
            $response = json_decode(file_get_contents($getChatMemberUrl), TRUE);
            if (isset($response['result']['status']) && $response['result']['status'] != 'left') {
                return true;
            }
        }
        return false;
    }

    private function handleNotMember() {
        $this->sendVoice(
            $this->chatId,
            'https://sadataalaee.com/bots/feedback/مومنتوم چیست؟.mp3'
        );
        $this->sendMessage($this->chatId, NOT_MEMBER_MESSAGE);
    }

    private function getUserInfo() {
        $userInfo = "👤 اطلاعات کاربر:\n";
        $userInfo .= "🔹 نام: {$this->firstName} {$this->lastName}\n";
        $userInfo .= "🔹 نام کاربری: {$this->username}\n";
        $userInfo .= "🔹 شناسه کاربری: {$this->userId}\n";
        return $userInfo;
    }

    private function downloadAndSaveTelegramFile($fileId, $type) {
        // گرفتن file_path از تلگرام
        $getFileUrl = "https://api.telegram.org/bot{$this->apiToken}/getFile?file_id={$fileId}";
        $fileInfo = json_decode(file_get_contents($getFileUrl), true);
        if (!isset($fileInfo['result']['file_path'])) return null;

        $filePath = $fileInfo['result']['file_path'];
        $ext = pathinfo($filePath, PATHINFO_EXTENSION);
        $saveDir = __DIR__ . '/uploads/feedback/';
        if (!is_dir($saveDir)) mkdir($saveDir, 0777, true);

        $filename = $type . '_' . $this->userId . '_' . time() . '.' . $ext;
        $savePath = $saveDir . $filename;

        $downloadUrl = "https://api.telegram.org/file/bot{$this->apiToken}/{$filePath}";
        file_put_contents($savePath, file_get_contents($downloadUrl));

        // مسیر نسبی برای ذخیره در دیتابیس
        return 'uploads/feedback/' . $filename;
    }

    private function processFeedback() {
        $userInfo = $this->getUserInfo();
        $name = $this->firstName . ' ' . $this->lastName;
        $username = $this->username;
        $user_id = $this->userId;
        $text = null;
        $voice = null;
        $video = null;
        $photo = null;
        $channelsInfo = $this->getUserChannelsInfo();
        $channel_count = $channelsInfo['count'];
        $channel_names = $channelsInfo['names'];

        if (isset($this->message['photo'])) {
            $photoArray = $this->message['photo'];
            $photoId = end($photoArray)['file_id'];
            $photo = $this->downloadAndSaveTelegramFile($photoId, 'photo');
            $caption = isset($this->message['caption']) ? $this->message['caption'] : null;
            $text = $caption;
            $this->handlePhoto($userInfo, $channel_count, $channel_names);
        } elseif (isset($this->message['video'])) {
            $videoId = $this->message['video']['file_id'];
            $video = $this->downloadAndSaveTelegramFile($videoId, 'video');
            $caption = isset($this->message['caption']) ? $this->message['caption'] : null;
            $text = $caption;
            $this->handleVideo($userInfo, $channel_count, $channel_names);
        } elseif (isset($this->message['document'])) {
            $this->handleDocument($userInfo, $channel_count, $channel_names);
        } elseif (isset($this->message['voice'])) {
            $voiceId = $this->message['voice']['file_id'];
            $voice = $this->downloadAndSaveTelegramFile($voiceId, 'voice');
            $caption = isset($this->message['caption']) ? $this->message['caption'] : null;
            $text = $caption;
            $this->handleVoice($userInfo, $channel_count, $channel_names);
        } elseif (isset($this->message['text'])) {
            $text = $this->text;
            $this->handleText($userInfo, $channel_count, $channel_names);
        }

        // ذخیره فیدبک در دیتابیس
        try {
            $this->db->addFeedback($name, $username, $user_id, $text, $voice, $video, $photo, $channel_count);
        } catch (Exception $e) {
            error_log($e->getMessage());
        }

        $this->sendMessage($this->chatId, CONFIRMATION_MESSAGE);
    }

    private function getUserChannelsInfo() {
        global $channels;
        $count = 0;
        $names = [];
        foreach ($channels as $channelId => $channelName) {
            $getChatMemberUrl = "https://api.telegram.org/bot{$this->apiToken}/getChatMember?chat_id=$channelId&user_id={$this->userId}";
            $response = json_decode(file_get_contents($getChatMemberUrl), TRUE);
            if (isset($response['result']['status']) && $response['result']['status'] != 'left') {
                $count++;
                $names[] = $channelName;
            }
        }
        return ['count' => $count, 'names' => $names];
    }

    private function handlePhoto($userInfo, $channel_count = null, $channel_names = null) {
        $photoArray = $this->message['photo'];
        $fileId = end($photoArray)['file_id'];
        $caption = isset($this->message['caption']) ? $this->message['caption'] : "بدون کپشن";
        $message = $userInfo . "\n📸 عکس:\n" . $caption;
        if ($channel_count !== null) {
            $message .= "\n📢 تعداد کانال‌های عضو: $channel_count";
            if (!empty($channel_names)) {
                $message .= "\nکانال‌ها: " . implode('، ', $channel_names);
            }
        }
        $this->sendPhoto(FEEDBACK_CHAT_ID, $fileId, $message);
        $this->sendPhoto(FEEDBACK_CHAT_ID_2, $fileId, $message);
    }

    private function handleVideo($userInfo, $channel_count = null, $channel_names = null) {
        $fileId = $this->message['video']['file_id'];
        $caption = isset($this->message['caption']) ? $this->message['caption'] : "بدون کپشن";
        $message = $userInfo . "\n🎥 ویدیو:\n" . $caption;
        if ($channel_count !== null) {
            $message .= "\n📢 تعداد کانال‌های عضو: $channel_count";
            if (!empty($channel_names)) {
                $message .= "\nکانال‌ها: " . implode('، ', $channel_names);
            }
        }
        $this->sendVideo(FEEDBACK_CHAT_ID, $fileId, $message);
        $this->sendVideo(FEEDBACK_CHAT_ID_2, $fileId, $message);
    }

    private function handleDocument($userInfo, $channel_count = null, $channel_names = null) {
        $fileId = $this->message['document']['file_id'];
        $caption = isset($this->message['caption']) ? $this->message['caption'] : "بدون کپشن";
        $message = $userInfo . "\n📄 فایل:\n" . $caption;
        if ($channel_count !== null) {
            $message .= "\n📢 تعداد کانال‌های عضو: $channel_count";
            if (!empty($channel_names)) {
                $message .= "\nکانال‌ها: " . implode('، ', $channel_names);
            }
        }
        $this->sendDocument(FEEDBACK_CHAT_ID, $fileId, $message);
        $this->sendDocument(FEEDBACK_CHAT_ID_2, $fileId, $message);
    }

    private function handleVoice($userInfo, $channel_count = null, $channel_names = null) {
        $fileId = $this->message['voice']['file_id'];
        $caption = isset($this->message['caption']) ? $this->message['caption'] : "بدون کپشن";
        $message = $userInfo . "\n🎙 ویس:\n" . $caption;
        if ($channel_count !== null) {
            $message .= "\n📢 تعداد کانال‌های عضو: $channel_count";
            if (!empty($channel_names)) {
                $message .= "\nکانال‌ها: " . implode('، ', $channel_names);
            }
        }
        $this->sendVoice(FEEDBACK_CHAT_ID, $fileId, $message);
        $this->sendVoice(FEEDBACK_CHAT_ID_2, $fileId, $message);
    }

    private function handleText($userInfo, $channel_count = null, $channel_names = null) {
        $message = $userInfo . "\n📝 فیدبک:\n" . $this->text;
        if ($channel_count !== null) {
            $message .= "\n📢 تعداد کانال‌های عضو: $channel_count";
            if (!empty($channel_names)) {
                $message .= "\nکانال‌ها: " . implode('، ', $channel_names);
            }
        }
        $this->sendMessage(FEEDBACK_CHAT_ID, $message);
        $this->sendMessage(FEEDBACK_CHAT_ID_2, $message);
    }

    // متدهای ارسال پیام
    private function sendMessage($chatId, $text, $replyMarkup = null) {
        $url = "https://api.telegram.org/bot{$this->apiToken}/sendMessage";
        $params = [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'HTML'
        ];
        
        if ($replyMarkup) {
            $params['reply_markup'] = $replyMarkup;
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_exec($ch);
        curl_close($ch);
    }

    private function sendPhoto($chatId, $photo, $caption) {
        file_get_contents("https://api.telegram.org/bot{$this->apiToken}/sendPhoto?chat_id=$chatId&photo=$photo&caption=" . urlencode($caption));
    }

    private function sendVideo($chatId, $video, $caption) {
        file_get_contents("https://api.telegram.org/bot{$this->apiToken}/sendVideo?chat_id=$chatId&video=$video&caption=" . urlencode($caption));
    }

    private function sendDocument($chatId, $filePath, $caption = null) {
        $url = "https://api.telegram.org/bot{$this->apiToken}/sendDocument";
        $postFields = [
            'chat_id' => $chatId,
            'document' => new CURLFile(realpath($filePath)),
        ];
        if ($caption) {
            $postFields['caption'] = $caption;
        }
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST,1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_exec($ch);
        curl_close($ch);
    }

    private function sendVoice($chatId, $voice, $caption = '') {
        file_get_contents("https://api.telegram.org/bot{$this->apiToken}/sendVoice?chat_id=$chatId&voice=$voice&caption=" . urlencode($caption));
    }

    private function handleCallback($callback) {
        $data = $callback['data'];
        $chatId = $callback['message']['chat']['id'];
        $messageId = $callback['message']['message_id'];

        if (strpos($data, 'delete_') === 0) {
            $confId = intval(str_replace('delete_', '', $data));
            try {
                if ($this->db->deleteConference($confId)) {
                    $this->editMessageText($chatId, $messageId, "✅ همایش با موفقیت حذف شد.");
                } else {
                    $this->editMessageText($chatId, $messageId, "❌ خطا در حذف همایش.");
                }
            } catch (Exception $e) {
                $this->editMessageText($chatId, $messageId, "❌ خطا در حذف همایش: " . $e->getMessage());
            }
        } elseif (strpos($data, 'confinfo_') === 0) {
            $confId = intval(str_replace('confinfo_', '', $data));
            $conf = $this->db->getConferenceById($confId);
            if ($conf) {
                // بررسی ثبت‌نام کاربر
                $alreadyRegistered = $this->db->isUserRegistered($confId, $this->userId);
                $msg = "🔹 نام: {$conf['name']}\n";
                $msg .= "👥 تعداد نفرات: {$conf['participants_count']}\n";
                $msg .= "📅 تاریخ: {$conf['conference_date']}\n";
                $msg .= "⏰ ساعت: {$conf['start_time']} تا {$conf['end_time']}\n";
                $msg .= "📍 آدرس: {$conf['address']}\n";
                $msg .= "🔗 لینک لوکیشن: {$conf['location_link']}\n";
                $msg .= "📢 کانال: {$conf['channel_id']}\n";
                if ($alreadyRegistered) {
                    $msg .= "\n✅ <b>ثبت‌نام شما با موفقیت انجام شده است.</b>";
                    $this->editMessageText($chatId, $messageId, $msg);
                } else {
                    $keyboard = [
                        'inline_keyboard' => [
                            [
                                ['text' => 'ثبت‌نام در همایش', 'callback_data' => 'register_' . $conf['id']]
                            ]
                        ]
                    ];
                    $this->editMessageText($chatId, $messageId, $msg, $keyboard);
                }
            }
        } elseif (strpos($data, 'register_') === 0) {
            $confId = intval(str_replace('register_', '', $data));
            $this->startUserRegister($confId, $chatId, $messageId);
        } elseif (strpos($data, 'export_') === 0) {
            $confId = intval(str_replace('export_', '', $data));
            $participants = $this->db->getParticipants($confId);
            if (empty($participants)) {
                $this->sendMessage($chatId, "❌ هیچ ثبت‌نامی برای این همایش وجود ندارد.");
                return;
            }
            $fileName = 'participants_' . $confId . '_' . time() . '.csv';
            $fp = fopen($fileName, 'w');
            // UTF-8 BOM برای نمایش درست فارسی در اکسل
            fwrite($fp, chr(0xEF).chr(0xBB).chr(0xBF));
            fputcsv($fp, ['نام', 'نام خانوادگی', 'شماره تلفن', 'تاریخ ثبت‌نام']);
            foreach ($participants as $p) {
                fputcsv($fp, [$p['first_name'], $p['last_name'], $p['phone'], $p['register_date']]);
            }
            fclose($fp);
            $this->sendDocument($chatId, $fileName, "📥 خروجی اکسل ثبت‌نامی‌های همایش (CSV)");
            unlink($fileName);
            return;
        }
    }

    private function editMessageText($chatId, $messageId, $text, $keyboard = null) {
        $url = "https://api.telegram.org/bot{$this->apiToken}/editMessageText";
        $params = [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $text,
            'parse_mode' => 'HTML'
        ];
        if ($keyboard) {
            $params['reply_markup'] = json_encode($keyboard);
        }
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_exec($ch);
        curl_close($ch);
    }

    private function startUserRegister($confId, $chatId, $messageId) {
        $state = [
            'state' => 'REGISTER',
            'step' => 'first_name',
            'data' => [
                'conference_id' => $confId,
                'user_id' => $this->userId
            ]
        ];
        $this->setUserRegisterState($state);
        $this->sendMessage($chatId, "لطفاً نام خود را وارد کنید:");
    }

    private function setUserRegisterState($state) {
        file_put_contents('register_state_' . $this->userId . '.json', json_encode($state));
    }
    private function getUserRegisterState() {
        $file = 'register_state_' . $this->userId . '.json';
        if (file_exists($file)) {
            $json = file_get_contents($file);
            return json_decode($json, true);
        }
        return null;
    }
    private function clearUserRegisterState() {
        $file = 'register_state_' . $this->userId . '.json';
        if (file_exists($file)) unlink($file);
    }
    private function checkUserRegisterState() {
        $state = $this->getUserRegisterState();
        return $state && $state['state'] === 'REGISTER';
    }
    private function handleUserRegisterState() {
        $state = $this->getUserRegisterState();
        $data = $state['data'];
        switch ($state['step']) {
            case 'first_name':
                $data['first_name'] = $this->faToEn($this->text);
                $state['step'] = 'last_name';
                $state['data'] = $data;
                $this->setUserRegisterState($state);
                $this->sendMessage($this->chatId, "لطفاً نام خانوادگی خود را وارد کنید:");
                return;
            case 'last_name':
                $data['last_name'] = $this->faToEn($this->text);
                $state['step'] = 'phone';
                $state['data'] = $data;
                $this->setUserRegisterState($state);
                $this->sendMessage($this->chatId, "لطفاً شماره تلفن خود را وارد کنید:");
                return;
            case 'phone':
                $data['phone'] = $this->faToEn($this->text);
                // ذخیره در دیتابیس
                try {
                    $this->db->addParticipant(
                        $data['conference_id'],
                        $data['user_id'],
                        $data['first_name'],
                        $data['last_name'],
                        $data['phone']
                    );
                    $this->db->decreaseParticipantsCount($data['conference_id']);
                    $this->sendMessage($this->chatId, "✅ ثبت‌نام شما با موفقیت انجام شد.");
                } catch (Exception $e) {
                    $this->sendMessage($this->chatId, "❌ خطا در ثبت‌نام: " . $e->getMessage());
                }
                $this->clearUserRegisterState();
                $this->showUserMenu();
                return;
        }
    }

    private function showUserConferences() {
        $conferences = $this->db->getConferences();
        if (!$conferences) return;
        $userChannels = $this->getUserChannelsInfo();
        $userChannelIds = [];
        global $channels;
        foreach ($channels as $id => $name) {
            if (in_array($name, $userChannels['names'])) {
                $userChannelIds[] = $id;
            }
        }
        $buttons = [];
        foreach ($conferences as $conf) {
            if (in_array($conf['channel_id'], $userChannelIds)) {
                $buttons[] = [['text' => $conf['name'], 'callback_data' => 'confinfo_' . $conf['id']]];
            }
        }
        if (!empty($buttons)) {
            $keyboard = ['inline_keyboard' => $buttons];
            $this->sendMessage($this->chatId, "برای ثبت‌نام در همایش، روی نام همایش بزنید:", json_encode($keyboard));
        }
    }

    private function showUserMenu() {
        $keyboard = [
            'keyboard' => [
                [['text' => 'ارسال فیدبک'], ['text' => 'لیست همایش‌ها']]
            ],
            'resize_keyboard' => true
        ];
        $this->sendMessage(
            $this->chatId,
            "👋 خوش آمدید! لطفاً یکی از گزینه‌های زیر را انتخاب کنید:",
            json_encode($keyboard)
        );
    }

    // مدیریت وضعیت ارسال فیدبک
    private function setUserFeedbackState() {
        file_put_contents('feedback_state_' . $this->userId . '.json', json_encode(['state' => 'FEEDBACK']));
    }
    private function getUserFeedbackState() {
        $file = 'feedback_state_' . $this->userId . '.json';
        if (file_exists($file)) {
            $json = file_get_contents($file);
            return json_decode($json, true);
        }
        return null;
    }
    private function clearUserFeedbackState() {
        $file = 'feedback_state_' . $this->userId . '.json';
        if (file_exists($file)) unlink($file);
    }
    private function checkUserFeedbackState() {
        $state = $this->getUserFeedbackState();
        return $state && $state['state'] === 'FEEDBACK';
    }
    private function handleUserFeedbackState() {
        $this->processFeedback();
        $this->clearUserFeedbackState();
        $this->showUserMenu();
    }

    // تابع کمکی تبدیل اعداد فارسی به انگلیسی
    private function faToEn($string) {
        $fa = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹','٫','،'];
        $en = ['0','1','2','3','4','5','6','7','8','9','.',''];
        return str_replace($fa, $en, $string);
    }
}

// اجرای ربات
$bot = new TelegramBot();
$bot->handleUpdate(); 