<?php
class Database {
    private $conn;
    private $host = 'localhost';
    private $dbname = 'berandco_momentum';
    private $username = 'berandco_bot';
    private $password = 'Yw$@GjGscJhW';

    public function __construct() {
        try {
            $this->conn = new mysqli($this->host, $this->username, $this->password, $this->dbname);
            
            if ($this->conn->connect_error) {
                throw new Exception("خطا در اتصال به دیتابیس: " . $this->conn->connect_error);
            }
            
            $this->conn->set_charset("utf8mb4");
        } catch (Exception $e) {
            error_log($e->getMessage());
            throw new Exception("خطا در اتصال به دیتابیس");
        }
    }

    public function createTables() {
        try {
            if (!$this->conn) {
                throw new Exception("اتصال به دیتابیس برقرار نشده است");
            }

            // جدول همایش‌ها
            $this->conn->multi_query("CREATE TABLE IF NOT EXISTS conferences (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(255) NOT NULL,
                participants_count INT NOT NULL,
                conference_date DATE NOT NULL,
                address TEXT NOT NULL,
                location_link TEXT NOT NULL,
                start_time TIME NOT NULL,
                end_time TIME NOT NULL,
                channel_id VARCHAR(255) NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            ) CHARACTER SET utf8mb4 COLLATE utf8mb4_persian_ci");

            // جدول شرکت‌کنندگان
            $this->conn->multi_query("CREATE TABLE IF NOT EXISTS participants (
                id INT AUTO_INCREMENT PRIMARY KEY,
                first_name VARCHAR(255) NOT NULL,
                last_name VARCHAR(255) NOT NULL,
                phone VARCHAR(20) NOT NULL,
                register_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                conference_id INT NOT NULL,
                attendance_status TINYINT(1) DEFAULT 0,
                FOREIGN KEY (conference_id) REFERENCES conferences(id)
            ) CHARACTER SET utf8mb4 COLLATE utf8mb4_persian_ci");

            // جدول فیدبک‌ها
            $this->conn->multi_query("CREATE TABLE IF NOT EXISTS feedbacks (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(255),
                username VARCHAR(255),
                user_id BIGINT,
                text TEXT,
                voice TEXT,
                video TEXT,
                photo TEXT,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                channel_count INT
            ) CHARACTER SET utf8mb4 COLLATE utf8mb4_persian_ci");

            // جدول به‌روزرسانی‌های کاربران
            $this->conn->multi_query("CREATE TABLE IF NOT EXISTS user_updates (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id BIGINT NOT NULL,
                update_type VARCHAR(50) NOT NULL,
                update_data TEXT,
                is_sent TINYINT(1) DEFAULT 0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                sent_at TIMESTAMP NULL,
                UNIQUE KEY unique_user_update (user_id, update_type)
            ) CHARACTER SET utf8mb4 COLLATE utf8mb4_persian_ci");

            return true;
        } catch(Exception $e) {
            error_log("خطا در ایجاد جداول: " . $e->getMessage());
            throw new Exception("خطا در ایجاد جداول");
        }
    }

    public function getConferences() {
        try {
            $sql = "SELECT * FROM conferences ORDER BY conference_date DESC";
            $result = $this->conn->query($sql);
            
            if (!$result) {
                throw new Exception("خطا در دریافت لیست کنفرانس‌ها");
            }
            
            $conferences = [];
            while ($row = $result->fetch_assoc()) {
                $conferences[] = $row;
            }
            return $conferences;
        } catch (Exception $e) {
            error_log($e->getMessage());
            throw new Exception("خطا در دریافت لیست کنفرانس‌ها");
        }
    }

    public function addConference($data) {
        try {
            $sql = "INSERT INTO conferences (name, participants_count, conference_date, address, location_link, start_time, end_time, channel_id) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = $this->conn->prepare($sql);
            
            if (!$stmt) {
                error_log($this->conn->error);
                throw new Exception("خطا در آماده‌سازی دستور SQL");
            }
            
            $stmt->bind_param("sissssss", 
                $data['name'],
                $data['participants_count'],
                $data['conference_date'],
                $data['address'],
                $data['location_link'],
                $data['start_time'],
                $data['end_time'],
                $data['channel_id']
            );
            
            if (!$stmt->execute()) {
                error_log($stmt->error);
                throw new Exception("خطا در ثبت همایش");
            }
            
            return $this->conn->insert_id;
        } catch (Exception $e) {
            error_log($e->getMessage());
            throw new Exception("خطا در ثبت همایش");
        }
    }

    public function deleteConference($id) {
        try {
            $sql = "DELETE FROM conferences WHERE id = ?";
            $stmt = $this->conn->prepare($sql);
            
            if (!$stmt) {
                throw new Exception("خطا در آماده‌سازی دستور SQL");
            }
            
            $stmt->bind_param("i", $id);
            
            if (!$stmt->execute()) {
                throw new Exception("خطا در حذف کنفرانس");
            }
            
            return true;
        } catch (Exception $e) {
            error_log($e->getMessage());
            throw new Exception("خطا در حذف کنفرانس");
        }
    }

    public function addParticipant($conference_id, $user_id, $first_name, $last_name, $phone) {
        try {
            $sql = "INSERT INTO participants (conference_id, user_id, first_name, last_name, phone) VALUES (?, ?, ?, ?, ?)";
            $stmt = $this->conn->prepare($sql);
            if (!$stmt) {
                error_log($this->conn->error);
                throw new Exception("خطا در آماده‌سازی دستور SQL");
            }
            $stmt->bind_param("iisss", $conference_id, $user_id, $first_name, $last_name, $phone);
            if (!$stmt->execute()) {
                error_log($stmt->error);
                throw new Exception("خطا در ثبت شرکت‌کننده");
            }
            return true;
        } catch (Exception $e) {
            error_log($e->getMessage());
            throw new Exception("خطا در ثبت شرکت‌کننده");
        }
    }

    public function getParticipants($conference_id) {
        try {
            $sql = "SELECT id, first_name, last_name, phone, register_date FROM participants WHERE conference_id = ?";
            $stmt = $this->conn->prepare($sql);
            if (!$stmt) {
                throw new Exception("خطا در آماده‌سازی دستور SQL");
            }
            $stmt->bind_param("i", $conference_id);
            if (!$stmt->execute()) {
                throw new Exception("خطا در دریافت لیست شرکت‌کنندگان");
            }
            $stmt->store_result();
            $stmt->bind_result($id, $first_name, $last_name, $phone, $register_date);
            $participants = [];
            while ($stmt->fetch()) {
                $participants[] = [
                    'id' => $id,
                    'first_name' => $first_name,
                    'last_name' => $last_name,
                    'phone' => $phone,
                    'register_date' => $register_date
                ];
            }
            $stmt->close();
            return $participants;
        } catch (Exception $e) {
            error_log($e->getMessage());
            throw new Exception("خطا در دریافت لیست شرکت‌کنندگان");
        }
    }

    public function isManager($user_id) {
        try {
            $sql = "SELECT * FROM managers WHERE user_id = ?";
            $stmt = $this->conn->prepare($sql);
            
            if (!$stmt) {
                throw new Exception("خطا در آماده‌سازی دستور SQL");
            }
            
            $stmt->bind_param("i", $user_id);
            
            if (!$stmt->execute()) {
                throw new Exception("خطا در بررسی دسترسی مدیر");
            }
            
            $result = $stmt->get_result();
            return $result->num_rows > 0;
        } catch (Exception $e) {
            error_log($e->getMessage());
            throw new Exception("خطا در بررسی دسترسی مدیر");
        }
    }

    public function addFeedback($name, $username, $user_id, $text, $voice, $video, $photo, $channel_count) {
        try {
            $sql = "INSERT INTO feedbacks (name, username, user_id, text, voice, video, photo, channel_count) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = $this->conn->prepare($sql);
            if (!$stmt) {
                error_log($this->conn->error);
                throw new Exception("خطا در آماده‌سازی دستور SQL");
            }
            $stmt->bind_param("ssissssi", $name, $username, $user_id, $text, $voice, $video, $photo, $channel_count);
            if (!$stmt->execute()) {
                error_log($stmt->error);
                throw new Exception("خطا در ثبت فیدبک");
            }
            return $this->conn->insert_id;
        } catch (Exception $e) {
            error_log($e->getMessage());
            throw new Exception("خطا در ثبت فیدبک");
        }
    }

    public function __destruct() {
        if ($this->conn) {
            $this->conn->close();
        }
    }
    public function getConferenceById($id) {
    try {
        $sql = "SELECT * FROM conferences WHERE id = ?";
        $stmt = $this->conn->prepare($sql);
        if (!$stmt) {
            throw new Exception("خطا در آماده‌سازی دستور SQL");
        }
        $stmt->bind_param("i", $id);
        if (!$stmt->execute()) {
            throw new Exception("خطا در دریافت اطلاعات همایش");
        }
        $result = [];
        $stmt->store_result();
        $meta = $stmt->result_metadata();
        if ($meta) {
            $fields = [];
            $row = [];
            while ($field = $meta->fetch_field()) {
                $fields[] = &$row[$field->name];
            }
            call_user_func_array([$stmt, 'bind_result'], $fields);
            if ($stmt->fetch()) {
                foreach ($row as $key => $val) {
                    $result[$key] = $val;
                }
            }
        }
        $stmt->close();
        return $result;
    } catch (Exception $e) {
        error_log($e->getMessage());
        return null;
    }
}

public function isUserRegistered($conference_id, $user_id) {
    try {
        $sql = "SELECT id FROM participants WHERE conference_id = ? AND user_id = ?";
        $stmt = $this->conn->prepare($sql);
        if (!$stmt) return false;
        $stmt->bind_param("ii", $conference_id, $user_id);
        $stmt->execute();
        $stmt->store_result();
        return $stmt->num_rows > 0;
    } catch (Exception $e) {
        return false;
    }
}

public function decreaseParticipantsCount($conference_id) {
    try {
        $sql = "UPDATE conferences SET participants_count = participants_count - 1 WHERE id = ? AND participants_count > 0";
        $stmt = $this->conn->prepare($sql);
        if (!$stmt) return false;
        $stmt->bind_param("i", $conference_id);
        return $stmt->execute();
    } catch (Exception $e) {
        return false;
    }
}

// متدهای مدیریت به‌روزرسانی‌ها
public function addUserUpdate($user_id, $update_type, $update_data) {
    try {
        $sql = "INSERT INTO user_updates (user_id, update_type, update_data) 
                VALUES (?, ?, ?) 
                ON DUPLICATE KEY UPDATE 
                update_data = VALUES(update_data),
                is_sent = 0,
                sent_at = NULL";
        
        $stmt = $this->conn->prepare($sql);
        if (!$stmt) {
            throw new Exception("خطا در آماده‌سازی دستور SQL");
        }
        
        $stmt->bind_param("iss", $user_id, $update_type, $update_data);
        if (!$stmt->execute()) {
            throw new Exception("خطا در ثبت به‌روزرسانی");
        }
        
        return true;
    } catch (Exception $e) {
        error_log($e->getMessage());
        throw new Exception("خطا در ثبت به‌روزرسانی");
    }
}

public function getPendingUpdates() {
    try {
        $sql = "SELECT * FROM user_updates WHERE is_sent = 0 ORDER BY created_at ASC";
        $result = $this->conn->query($sql);
        
        if (!$result) {
            throw new Exception("خطا در دریافت به‌روزرسانی‌های در انتظار");
        }
        
        $updates = [];
        while ($row = $result->fetch_assoc()) {
            $updates[] = $row;
        }
        return $updates;
    } catch (Exception $e) {
        error_log($e->getMessage());
        throw new Exception("خطا در دریافت به‌روزرسانی‌های در انتظار");
    }
}

public function markUpdateAsSent($update_id) {
    try {
        $sql = "UPDATE user_updates SET is_sent = 1, sent_at = CURRENT_TIMESTAMP WHERE id = ?";
        $stmt = $this->conn->prepare($sql);
        
        if (!$stmt) {
            throw new Exception("خطا در آماده‌سازی دستور SQL");
        }
        
        $stmt->bind_param("i", $update_id);
        if (!$stmt->execute()) {
            throw new Exception("خطا در به‌روزرسانی وضعیت");
        }
        
        return true;
    } catch (Exception $e) {
        error_log($e->getMessage());
        throw new Exception("خطا در به‌روزرسانی وضعیت");
    }
}

public function getAllUsers() {
    try {
        $sql = "SELECT DISTINCT user_id FROM participants UNION SELECT DISTINCT user_id FROM feedbacks";
        $result = $this->conn->query($sql);
        
        if (!$result) {
            throw new Exception("خطا در دریافت لیست کاربران");
        }
        
        $users = [];
        while ($row = $result->fetch_assoc()) {
            $users[] = $row['user_id'];
        }
        return $users;
    } catch (Exception $e) {
        error_log($e->getMessage());
        throw new Exception("خطا در دریافت لیست کاربران");
    }
}
} 

