<?php
require_once 'config.php';
require_once 'database.php';

class UpdateProcessor {
    private $db;
    private $apiToken;
    private $updateTypes = [
        'NEW_CONFERENCE' => 'همایش جدیدی اضافه شد',
        'CONFERENCE_UPDATE' => 'اطلاعات همایش به‌روزرسانی شد',
        'CONFERENCE_CANCELLED' => 'همایش لغو شد',
        'GENERAL_ANNOUNCEMENT' => 'اطلاعیه عمومی'
    ];

    public function __construct() {
        $this->db = new Database();
        $this->apiToken = BOT_TOKEN;
    }

    public function processUpdates() {
        try {
            $updates = $this->db->getPendingUpdates();
            
            foreach ($updates as $update) {
                $this->sendUpdate($update);
                $this->db->markUpdateAsSent($update['id']);
                
                // تاخیر کوتاه برای جلوگیری از محدودیت‌های API تلگرام
                sleep(1);
            }
            
            return true;
        } catch (Exception $e) {
            error_log("خطا در پردازش به‌روزرسانی‌ها: " . $e->getMessage());
            return false;
        }
    }

    private function sendUpdate($update) {
        $message = $this->formatUpdateMessage($update);
        $this->sendTelegramMessage($update['user_id'], $message);
    }

    private function formatUpdateMessage($update) {
        $type = $this->updateTypes[$update['update_type']] ?? $update['update_type'];
        $data = json_decode($update['update_data'], true);
        
        $message = "🔔 {$type}\n\n";
        
        switch ($update['update_type']) {
            case 'NEW_CONFERENCE':
            case 'CONFERENCE_UPDATE':
                $message .= "🔹 نام: {$data['name']}\n";
                $message .= "📅 تاریخ: {$data['conference_date']}\n";
                $message .= "⏰ ساعت: {$data['start_time']} تا {$data['end_time']}\n";
                $message .= "📍 آدرس: {$data['address']}\n";
                break;
                
            case 'CONFERENCE_CANCELLED':
                $message .= "🔹 نام همایش: {$data['name']}\n";
                $message .= "📅 تاریخ: {$data['conference_date']}\n";
                break;
                
            case 'GENERAL_ANNOUNCEMENT':
                $message .= $data['message'];
                break;
        }
        
        return $message;
    }

    private function sendTelegramMessage($chatId, $text) {
        $url = "https://api.telegram.org/bot{$this->apiToken}/sendMessage";
        $data = [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'HTML'
        ];
        
        $options = [
            'http' => [
                'method' => 'POST',
                'header' => "Content-Type: application/x-www-form-urlencoded\r\n",
                'content' => http_build_query($data)
            ]
        ];
        
        $context = stream_context_create($options);
        $result = file_get_contents($url, false, $context);
        
        if ($result === FALSE) {
            throw new Exception("خطا در ارسال پیام به تلگرام");
        }
        
        return true;
    }
}

// اجرای پردازش به‌روزرسانی‌ها
$processor = new UpdateProcessor();
$processor->processUpdates(); 